import os
import numpy as np
import pandas as pd
import plotly.graph_objects as go
from scipy.stats import norm
from svglib.svglib import svg2rlg
from reportlab.graphics import renderPDF

# Plot settings
width, height = 600, 400
margin = dict(l=20, r=20, t=20, b=20)
font = dict(family='Times New Roman', size=24, color='black')
grid = dict(showgrid=True, gridcolor='lightgray',
            zeroline=True, zerolinecolor='lightgray')
legend = dict(x=0.53, y=0.98)
xmin = -3
xmax = 3
color_sqrtn = 'limegreen'
color_n = 'darkgreen'

def plot_score_dist(sqrt_csv: str, n_csv: str, output_basename: str):
    # Load samples
    df_sqrt = pd.read_csv(sqrt_csv)
    df_n    = pd.read_csv(n_csv)
    vals_sqrt = df_sqrt['value'].values
    vals_n    = df_n['value'].values

    # Range for PDF curve
    xs = np.linspace(xmin, xmax, 200)

    fig = go.Figure()

    # Empirical histograms
    fig.add_trace(go.Histogram(
        x=vals_sqrt,
        histnorm='probability density',
        name='n=256, 1/√n-scaling',
        marker_color=color_sqrtn,
        opacity=0.5
        # marker=dict(
        #     # line_width=0,
        #     color=color_sqrtn,
        # )
    ))
    fig.add_trace(go.Histogram(
        x=vals_n,
        histnorm='probability density',
        name='n=256, 1/n-scaling',
        marker_color=color_n,
        opacity=0.5
        # marker=dict(
        #     # line_width=0,
        #     color=color_n,
        # )
    ))

    # Theoretical PDF for 1/√n (standard normal)
    y_pdf = norm.pdf(xs)
    fig.add_trace(go.Scatter(
        x=xs, y=y_pdf,
        mode='lines',
        name='∞-width, 1/√n-scaling',
        line=dict(color=color_sqrtn, width=2)
    ))

    # Theoretical line for 1/n (point mass at 0)
    fig.add_trace(go.Scatter(
        x=[0, 0], y=[0, 7],
        mode='lines',
        name='∞-width, 1/n-scaling',
        line=dict(color=color_n, width=2)
    ))

    fig.update_layout(
        font=font,
        width=width, height=height,
        margin=margin,
        plot_bgcolor='white', paper_bgcolor='white',
        xaxis=dict(
            range=[xmin, xmax],
            showgrid=grid['showgrid'], gridcolor=grid['gridcolor'],
            zeroline=True, zerolinecolor=grid['zerolinecolor']
        ),
        yaxis=dict(
            showgrid=grid['showgrid'], gridcolor=grid['gridcolor'],
            zeroline=True, zerolinecolor=grid['zerolinecolor']
        ),
        # bargap=0,
        legend=legend
    )

    # Save via SVG -> PDF
    svg_file = f"{output_basename}.svg"
    pdf_file = f"{output_basename}.pdf"
    fig.write_image(svg_file)
    drawing = svg2rlg(svg_file)
    try:
        renderPDF.drawToFile(drawing, pdf_file)
    except PermissionError:
        alt = pdf_file.replace('.pdf', '_new.pdf')
        renderPDF.drawToFile(drawing, alt)
        print(f"Permission denied for '{pdf_file}', saved to '{alt}'")
    finally:
        os.remove(svg_file)

if __name__ == '__main__':
    plot_score_dist('score_sqrt.csv', 'score_n.csv', 'score_dist')
